﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/chargeGrops")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ChargeGroupController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IChargeGroupService chargeGroupService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public ChargeGroupController(IChargeGroupService chargeGroupService, IAuditLogService auditLogServices)
        {
            this.chargeGroupService = chargeGroupService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ChargeGroupModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]ChargeGroupFilterModel model)
        {
            model = (ChargeGroupFilterModel)EmptyFilter.Handler(model);
            var beds = await this.chargeGroupService.FetchAsync(model);
            return beds == null ? this.ServerError() : this.Success(beds);
        }
      

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] ChargeGroupModel model)
        {
            model = (ChargeGroupModel)EmptyFilter.Handler(model);
            var response = await this.chargeGroupService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Charge Group has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            //var auditLogModel = new AuditLogModel
            //{
            //    AccountId = model.CreatedBy,
            //    LogTypeId = (int)LogTypes.Coupons,
            //    LogFrom = (int)AccountType.Administrator,
            //    LogDate = DateTime.UtcNow,
            //    LogDescription = $"{model.WardName} ward has been added."
            //};
            //await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Bed has been added successfully.");
        }


        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] ChargeGroupModel model)
        {
            model = (ChargeGroupModel)EmptyFilter.Handler(model);
            var response = await this.chargeGroupService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given charge group has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            //var auditLogModel = new AuditLogModel
            //{
            //    AccountId = model.ModifiedBy,
            //    LogTypeId = (int)LogTypes.Coupons,
            //    LogFrom = (int)AccountType.Administrator,
            //    LogDate = DateTime.UtcNow,
            //    LogDescription = $"{model.CouponCode} coupon has been updated."
            //};
            // await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Charge group has been updated successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ChargeGroupModel model)
        {
            try
            {
                model = (ChargeGroupModel)EmptyFilter.Handler(model);
                var response = await this.chargeGroupService.DeleteAsync(model.ChargeGroupId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                //var auditLogModel = new AuditLogModel
                //{
                //    AccountId = model.ModifiedBy,
                //    LogTypeId = (int)LogTypes.Specializations,
                //    LogFrom = (int)AccountType.Administrator,
                //    LogDate = DateTime.UtcNow,
                //    LogDescription = $"{model.BedNumber} specialization has been deleted."
                //};
                //await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success("charge group has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("charge group can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }


        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("chargeGroup")]
        [ProducesResponseType(typeof(List<ChargeGroupModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRoomAsync([FromBody] ChargeGroupFilterModel model)
        {
            model = (ChargeGroupFilterModel)EmptyFilter.Handler(model);
            var chargeGroups = await this.chargeGroupService.FetchChargeGroupAsync(model);
            return chargeGroups == null ? this.ServerError() : this.Success(chargeGroups);
        }

    }
}